"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Badge } from "@/components/ui/badge"
import { ArrowLeft, Eye, Plus, X } from "lucide-react"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Logo } from "@/components/logo"

export default function CreateAdPage() {
  const { user } = useAuth()
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [adData, setAdData] = useState({
    title: "",
    description: "",
    imageUrl: "",
    targetUrl: "",
    keywords: [] as string[],
    dailyLimit: 100,
    monthlyLimit: 1000,
    costPerClick: 0.5,
    costPerImpression: 0.1,
  })
  const [keywordInput, setKeywordInput] = useState("")

  useEffect(() => {
    if (!user) {
      router.push("/ads-center/auth")
    }
  }, [user, router])

  const addKeyword = () => {
    if (keywordInput.trim() && !adData.keywords.includes(keywordInput.trim())) {
      setAdData({
        ...adData,
        keywords: [...adData.keywords, keywordInput.trim()],
      })
      setKeywordInput("")
    }
  }

  const removeKeyword = (keyword: string) => {
    setAdData({
      ...adData,
      keywords: adData.keywords.filter((k) => k !== keyword),
    })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!user) return

    setLoading(true)
    try {
      const response = await fetch("/api/user-ads", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...adData,
          userId: user.id,
        }),
      })

      if (response.ok) {
        router.push("/ads-center")
      } else {
        console.error("Failed to create ad")
      }
    } catch (error) {
      console.error("Error creating ad:", error)
    } finally {
      setLoading(false)
    }
  }

  const getHostname = (url: string) => {
    try {
      if (!url || url.trim() === "") return "yourwebsite.com"
      const urlObj = new URL(url)
      return urlObj.hostname
    } catch {
      return "yourwebsite.com"
    }
  }

  const isValidUrl = (url: string) => {
    try {
      if (!url || url.trim() === "") return false
      new URL(url)
      return true
    } catch {
      return false
    }
  }

  if (!user) return null

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b border-border bg-card">
        <div className="container mx-auto px-4 py-4 flex items-center gap-4">
          <Button variant="ghost" onClick={() => router.back()}>
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back
          </Button>
          <Logo size="sm" />
          <div>
            <h1 className="text-xl font-semibold text-foreground">Create New Advertisement</h1>
            <p className="text-sm text-muted-foreground">Design your ad campaign with real-time preview</p>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Ad Creation Form */}
          <div>
            <Card>
              <CardHeader>
                <CardTitle>Ad Details</CardTitle>
                <CardDescription>Fill in your advertisement information</CardDescription>
              </CardHeader>
              <CardContent>
                <form onSubmit={handleSubmit} className="space-y-6">
                  <div className="space-y-2">
                    <Label htmlFor="title">Ad Title</Label>
                    <Input
                      id="title"
                      value={adData.title}
                      onChange={(e) => setAdData({ ...adData, title: e.target.value })}
                      placeholder="Enter compelling ad title"
                      required
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="description">Description</Label>
                    <Textarea
                      id="description"
                      value={adData.description}
                      onChange={(e) => setAdData({ ...adData, description: e.target.value })}
                      placeholder="Describe your product or service"
                      rows={3}
                      required
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="imageUrl">Image URL (Optional)</Label>
                    <Input
                      id="imageUrl"
                      type="url"
                      value={adData.imageUrl}
                      onChange={(e) => setAdData({ ...adData, imageUrl: e.target.value })}
                      placeholder="https://example.com/image.jpg"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="targetUrl">Target URL</Label>
                    <Input
                      id="targetUrl"
                      type="url"
                      value={adData.targetUrl}
                      onChange={(e) => setAdData({ ...adData, targetUrl: e.target.value })}
                      placeholder="https://yourwebsite.com"
                      required
                    />
                  </div>

                  <div className="space-y-2">
                    <Label>Keywords</Label>
                    <div className="flex gap-2">
                      <Input
                        value={keywordInput}
                        onChange={(e) => setKeywordInput(e.target.value)}
                        placeholder="Add keyword"
                        onKeyPress={(e) => e.key === "Enter" && (e.preventDefault(), addKeyword())}
                      />
                      <Button type="button" onClick={addKeyword} size="sm">
                        <Plus className="w-4 h-4" />
                      </Button>
                    </div>
                    <div className="flex flex-wrap gap-2 mt-2">
                      {adData.keywords.map((keyword) => (
                        <Badge key={keyword} variant="secondary" className="flex items-center gap-1">
                          {keyword}
                          <X className="w-3 h-3 cursor-pointer" onClick={() => removeKeyword(keyword)} />
                        </Badge>
                      ))}
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="dailyLimit">Daily Limit</Label>
                      <Input
                        id="dailyLimit"
                        type="number"
                        value={adData.dailyLimit}
                        onChange={(e) => setAdData({ ...adData, dailyLimit: Number.parseInt(e.target.value) })}
                        min="1"
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="monthlyLimit">Monthly Limit</Label>
                      <Input
                        id="monthlyLimit"
                        type="number"
                        value={adData.monthlyLimit}
                        onChange={(e) => setAdData({ ...adData, monthlyLimit: Number.parseInt(e.target.value) })}
                        min="1"
                        required
                      />
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="costPerClick">Cost per Click (₹)</Label>
                      <Input
                        id="costPerClick"
                        type="number"
                        step="0.01"
                        value={adData.costPerClick}
                        onChange={(e) => setAdData({ ...adData, costPerClick: Number.parseFloat(e.target.value) })}
                        min="0.01"
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="costPerImpression">Cost per Impression (₹)</Label>
                      <Input
                        id="costPerImpression"
                        type="number"
                        step="0.01"
                        value={adData.costPerImpression}
                        onChange={(e) => setAdData({ ...adData, costPerImpression: Number.parseFloat(e.target.value) })}
                        min="0.01"
                        required
                      />
                    </div>
                  </div>

                  <Button type="submit" className="w-full" disabled={loading}>
                    {loading ? "Creating Ad..." : "Create Advertisement"}
                  </Button>
                </form>
              </CardContent>
            </Card>
          </div>

          {/* Real-time Preview */}
          <div>
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Eye className="w-4 h-4" />
                  Live Preview
                </CardTitle>
                <CardDescription>See how your ad will appear in search results</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="border border-border rounded-lg p-4 bg-card">
                  <div className="flex items-start gap-3">
                    <div className="w-2 h-2 bg-orange-500 rounded-full mt-2 flex-shrink-0"></div>
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="text-xs bg-orange-100 text-orange-800 px-2 py-1 rounded font-medium">Ad</span>
                      </div>

                      {adData.imageUrl && isValidUrl(adData.imageUrl) && (
                        <img
                          src={adData.imageUrl || "/placeholder.svg"}
                          alt="Ad preview"
                          className="w-full h-32 object-cover rounded-md mb-3"
                          onError={(e) => {
                            e.currentTarget.style.display = "none"
                          }}
                        />
                      )}

                      <div className="flex items-center gap-2 mb-1">
                        <img
                          src={`https://www.google.com/s2/favicons?domain=${getHostname(adData.targetUrl)}`}
                          alt="Favicon"
                          className="w-4 h-4"
                          onError={(e) => {
                            e.currentTarget.src = "/default-favicon.png"
                          }}
                        />
                        <span className="text-sm text-green-600">{getHostname(adData.targetUrl)}</span>
                      </div>

                      <h3 className="text-lg font-medium text-blue-600 hover:underline cursor-pointer mb-1">
                        {adData.title || "Your Ad Title Here"}
                      </h3>

                      <p className="text-sm text-muted-foreground mb-2 line-clamp-2">
                        {adData.description || "Your compelling ad description will appear here..."}
                      </p>

                      {adData.keywords.length > 0 && (
                        <div className="flex flex-wrap gap-1">
                          {adData.keywords.slice(0, 3).map((keyword) => (
                            <Badge key={keyword} variant="outline" className="text-xs">
                              {keyword}
                            </Badge>
                          ))}
                        </div>
                      )}

                      <Button
                        variant="outline"
                        size="sm"
                        className="mt-3 text-blue-600 border-blue-600 hover:bg-blue-50 bg-transparent"
                      >
                        Visit Site
                      </Button>
                    </div>
                  </div>
                </div>

                <div className="mt-4 p-3 bg-muted rounded-lg">
                  <h4 className="font-medium text-sm mb-2">Estimated Costs</h4>
                  <div className="space-y-1 text-sm">
                    <div className="flex justify-between">
                      <span>Cost per Click:</span>
                      <span className="font-medium">₹{adData.costPerClick.toFixed(2)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Cost per Impression:</span>
                      <span className="font-medium">₹{adData.costPerImpression.toFixed(2)}</span>
                    </div>
                    <div className="flex justify-between text-primary font-medium pt-1 border-t">
                      <span>Daily Budget (est.):</span>
                      <span>₹{(adData.dailyLimit * adData.costPerClick * 0.1).toFixed(2)}</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  )
}
